@extends('layouts.dashboard.app')

@section('style')
<style>
/* ============================================
   MULTI-DOCUMENT UPLOAD WITH INDIVIDUAL TYPES
   ============================================ */

.upload-container {
    background: #fff;
    border-radius: 12px;
    box-shadow: 0 2px 12px rgba(0,0,0,0.08);
    padding: 30px;
    margin-top: 20px;
}

.upload-header {
    display: flex;
    align-items: center;
    justify-content: space-between;
    margin-bottom: 25px;
    padding-bottom: 20px;
    border-bottom: 1px solid #e5e7eb;
}

.upload-header h4 {
    margin: 0;
    color: #1f2937;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

.upload-header h4 i {
    color: #3b82f6;
}

/* Drop Zone */
.upload-zone {
    border: 2px dashed #d1d5db;
    border-radius: 12px;
    padding: 50px 20px;
    text-align: center;
    background: #f9fafb;
    transition: all 0.3s ease;
    cursor: pointer;
}

.upload-zone:hover,
.upload-zone.dragover {
    border-color: #3b82f6;
    background: #eff6ff;
}

.upload-zone i {
    font-size: 56px;
    color: #9ca3af;
    margin-bottom: 15px;
}

.upload-zone:hover i { color: #3b82f6; }

.upload-zone p {
    color: #6b7280;
    margin-bottom: 5px;
}

.upload-zone .formats {
    font-size: 12px;
    color: #9ca3af;
}

/* Files List */
.files-list {
    margin-top: 25px;
}

.files-list-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
    padding-bottom: 10px;
    border-bottom: 1px solid #e5e7eb;
}

.files-list-header h5 {
    margin: 0;
    font-size: 16px;
    color: #374151;
}

.file-count {
    background: #3b82f6;
    color: #fff;
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 13px;
    font-weight: 500;
}

/* File Item Card */
.file-item {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 15px;
    background: #f9fafb;
    border: 1px solid #e5e7eb;
    border-radius: 10px;
    margin-bottom: 12px;
    transition: all 0.2s;
}

.file-item:hover {
    border-color: #3b82f6;
    box-shadow: 0 2px 8px rgba(59, 130, 246, 0.1);
}

.file-thumb {
    width: 60px;
    height: 60px;
    border-radius: 8px;
    overflow: hidden;
    flex-shrink: 0;
    background: #e5e7eb;
    display: flex;
    align-items: center;
    justify-content: center;
}

.file-thumb img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.file-thumb .pdf-icon {
    font-size: 28px;
    color: #ef4444;
}

.file-info {
    flex: 1;
    min-width: 0;
}

.file-name {
    font-weight: 500;
    color: #1f2937;
    margin-bottom: 4px;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.file-size {
    font-size: 12px;
    color: #9ca3af;
}

.file-type-select {
    width: 200px;
    flex-shrink: 0;
}

.file-type-select select {
    width: 100%;
    padding: 10px 12px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    font-size: 14px;
    background: #fff;
    color: #374151;
    cursor: pointer;
}

.file-type-select select:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.file-type-select select.is-invalid {
    border-color: #ef4444;
}

.file-remove {
    width: 36px;
    height: 36px;
    border: none;
    background: #fee2e2;
    color: #ef4444;
    border-radius: 8px;
    cursor: pointer;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 18px;
    flex-shrink: 0;
    transition: all 0.2s;
}

.file-remove:hover {
    background: #ef4444;
    color: #fff;
}

/* Actions */
.upload-actions {
    margin-top: 25px;
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-upload {
    padding: 12px 30px;
    font-weight: 500;
    font-size: 15px;
}

/* Progress */
.upload-progress {
    display: none;
    margin-top: 20px;
}

.upload-progress.active {
    display: block;
}

/* Info Alert */
.info-alert {
    background: #eff6ff;
    border: 1px solid #bfdbfe;
    border-radius: 10px;
    padding: 14px 18px;
    margin-bottom: 25px;
    display: flex;
    align-items: center;
    gap: 10px;
    color: #1e40af;
    font-size: 14px;
}

.info-alert i {
    font-size: 20px;
}

/* Empty State */
.empty-state {
    text-align: center;
    padding: 40px;
    color: #9ca3af;
}

.empty-state i {
    font-size: 48px;
    margin-bottom: 10px;
}

/* Responsive */
@media (max-width: 768px) {
    .file-item {
        flex-wrap: wrap;
    }
    .file-type-select {
        width: 100%;
        order: 3;
        margin-top: 10px;
    }
}
</style>
@endsection

@section('content')
<div class="row">
    <div class="page-title-box">
        <div class="page-title-left">
            <a href="{{ route('Dossier.show', $dossier->id) }}" class="btn btn-outline-dark">
                <i class="uil uil-arrow-left me-1"></i> Retour au Dossier
            </a>
        </div>
    </div>
</div>

<div class="upload-container">
    <div class="upload-header">
        <h4><i class="uil uil-cloud-upload"></i> Télécharger des Documents</h4>
        <span class="badge bg-info">{{ $dossier->ref }}</span>
    </div>

    <div class="info-alert">
        <i class="uil uil-info-circle"></i>
        <span><strong>Formats acceptés:</strong> JPG, PNG, GIF, PDF (max 10MB). Sélectionnez un type pour chaque fichier avant de soumettre.</span>
    </div>

    <!-- Drop Zone -->
    <div class="upload-zone" id="uploadZone">
        <i class="uil uil-cloud-upload"></i>
        <p class="mb-1"><strong>Glissez vos fichiers ici</strong></p>
        <p class="mb-2">ou cliquez pour parcourir</p>
        <span class="formats">JPG, PNG, GIF, PDF</span>
        <input type="file" id="fileInput" class="d-none" accept="image/*,.pdf" multiple>
    </div>

    <!-- Files List -->
    <div class="files-list" id="filesList" style="display: none;">
        <div class="files-list-header">
            <h5><i class="uil uil-file-check-alt me-2"></i>Fichiers à télécharger</h5>
            <span class="file-count" id="fileCount">0 fichiers</span>
        </div>
        <div id="filesContainer"></div>
    </div>

    <!-- Validation Error -->
    @if ($errors->any())
        <div class="alert alert-danger mt-3">
            <ul class="mb-0">
                @foreach ($errors->all() as $error)
                    <li>{{ $error }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <!-- Progress -->
    <div class="upload-progress" id="uploadProgress">
        <div class="progress" style="height: 8px;">
            <div class="progress-bar progress-bar-striped progress-bar-animated bg-primary" role="progressbar" style="width: 0%"></div>
        </div>
        <p class="text-center mt-2 text-muted">Téléchargement en cours...</p>
    </div>

    <!-- Actions -->
    <div class="upload-actions">
        <button type="button" id="submitBtn" class="btn btn-primary btn-upload" disabled>
            <i class="uil uil-upload me-1"></i> Télécharger tout
        </button>
        <button type="button" id="clearBtn" class="btn btn-outline-secondary" style="display: none;">
            <i class="uil uil-trash me-1"></i> Tout effacer
        </button>
    </div>
</div>
@endsection

@section('script')
<script src="https://cdnjs.cloudflare.com/ajax/libs/pdf.js/2.2.228/pdf.min.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const uploadZone = document.getElementById('uploadZone');
    const fileInput = document.getElementById('fileInput');
    const filesList = document.getElementById('filesList');
    const filesContainer = document.getElementById('filesContainer');
    const fileCount = document.getElementById('fileCount');
    const submitBtn = document.getElementById('submitBtn');
    const clearBtn = document.getElementById('clearBtn');
    const uploadProgress = document.getElementById('uploadProgress');
    
    // Available document types
    const documentTypes = @json($types->map(fn($t) => ['id' => $t->id, 'type' => $t->type]));
    
    // File storage
    let files = []; // { file, type, preview, id }
    let fileIdCounter = 0;
    
    // Dossier ID
    const dossierId = {{ $id }};

    // Click to upload
    uploadZone.addEventListener('click', () => fileInput.click());

    // Drag and drop
    uploadZone.addEventListener('dragover', (e) => {
        e.preventDefault();
        uploadZone.classList.add('dragover');
    });

    uploadZone.addEventListener('dragleave', () => {
        uploadZone.classList.remove('dragover');
    });

    uploadZone.addEventListener('drop', (e) => {
        e.preventDefault();
        uploadZone.classList.remove('dragover');
        handleFiles(e.dataTransfer.files);
    });

    // File input change
    fileInput.addEventListener('change', (e) => {
        handleFiles(e.target.files);
        fileInput.value = ''; // Reset so same file can be added again
    });

    // Handle files
    function handleFiles(fileList) {
        const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'application/pdf'];
        
        for (let file of fileList) {
            // Check file type
            if (!allowedTypes.includes(file.type) && !file.name.toLowerCase().endsWith('.pdf')) {
                alert('Type de fichier non autorisé: ' + file.name);
                continue;
            }
            
            // Check file size (10MB max)
            if (file.size > 10 * 1024 * 1024) {
                alert('Fichier trop volumineux (max 10MB): ' + file.name);
                continue;
            }

            const fileObj = {
                id: fileIdCounter++,
                file: file,
                type: null, // No type selected yet
                isPdf: file.type === 'application/pdf' || file.name.toLowerCase().endsWith('.pdf')
            };
            
            files.push(fileObj);
            createFileItem(fileObj);
        }
        
        updateUI();
    }

    // Create file item UI
    function createFileItem(fileObj) {
        const div = document.createElement('div');
        div.className = 'file-item';
        div.dataset.id = fileObj.id;
        
        // Get file size formatted
        const sizeKB = Math.round(fileObj.file.size / 1024);
        const sizeStr = sizeKB > 1024 ? (sizeKB / 1024).toFixed(1) + ' MB' : sizeKB + ' KB';
        
        // Type options HTML
        let typeOptions = '<option value="">-- Sélectionner Type --</option>';
        documentTypes.forEach(t => {
            typeOptions += `<option value="${t.id}">${t.type}</option>`;
        });
        
        div.innerHTML = `
            <div class="file-thumb">
                ${fileObj.isPdf 
                    ? '<i class="uil uil-file-alt pdf-icon"></i>' 
                    : '<img src="" alt="Preview">'}
            </div>
            <div class="file-info">
                <div class="file-name" title="${fileObj.file.name}">${fileObj.file.name}</div>
                <div class="file-size">${sizeStr}</div>
            </div>
            <div class="file-type-select">
                <select data-id="${fileObj.id}" required>
                    ${typeOptions}
                </select>
            </div>
            <button type="button" class="file-remove" data-id="${fileObj.id}">
                <i class="uil uil-times"></i>
            </button>
        `;
        
        // Load image preview if not PDF
        if (!fileObj.isPdf) {
            const reader = new FileReader();
            reader.onload = (e) => {
                div.querySelector('img').src = e.target.result;
            };
            reader.readAsDataURL(fileObj.file);
        }
        
        // Type select change
        div.querySelector('select').addEventListener('change', (e) => {
            const id = parseInt(e.target.dataset.id);
            const f = files.find(x => x.id === id);
            if (f) {
                f.type = e.target.value ? parseInt(e.target.value) : null;
            }
            e.target.classList.toggle('is-invalid', !e.target.value);
            updateUI();
        });
        
        // Remove button
        div.querySelector('.file-remove').addEventListener('click', (e) => {
            const id = parseInt(e.currentTarget.dataset.id);
            files = files.filter(x => x.id !== id);
            div.remove();
            updateUI();
        });
        
        filesContainer.appendChild(div);
    }

    // Update UI
    function updateUI() {
        const hasFiles = files.length > 0;
        const allTyped = files.every(f => f.type !== null);
        
        filesList.style.display = hasFiles ? 'block' : 'none';
        clearBtn.style.display = hasFiles ? 'inline-block' : 'none';
        submitBtn.disabled = !(hasFiles && allTyped);
        
        const count = files.length;
        fileCount.textContent = count + ' fichier' + (count > 1 ? 's' : '');
    }

    // Clear all
    clearBtn.addEventListener('click', () => {
        files = [];
        filesContainer.innerHTML = '';
        updateUI();
    });

    // Submit
    submitBtn.addEventListener('click', async () => {
        if (files.length === 0) return;
        
        // Validate all have types
        let valid = true;
        files.forEach(f => {
            const select = document.querySelector(`select[data-id="${f.id}"]`);
            if (!f.type) {
                select.classList.add('is-invalid');
                valid = false;
            }
        });
        
        if (!valid) {
            alert('Veuillez sélectionner un type pour chaque fichier.');
            return;
        }
        
        // Show progress
        submitBtn.disabled = true;
        uploadProgress.classList.add('active');
        const progressBar = uploadProgress.querySelector('.progress-bar');
        
        let uploaded = 0;
        const total = files.length;
        
        // Upload files one by one (using existing API)
        for (const f of files) {
            try {
                const formData = new FormData();
                formData.append('file', f.file);
                formData.append('type', f.type);
                formData.append('id', dossierId);
                
                const response = await fetch('/Document/Image', {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}'
                    },
                    body: formData
                });
                
                if (!response.ok) throw new Error('Upload failed');
                
                uploaded++;
                progressBar.style.width = Math.round((uploaded / total) * 100) + '%';
                
            } catch (err) {
                console.error('Error uploading:', f.file.name, err);
            }
        }
        
        // Redirect back to dossier
        window.location.href = '{{ route("Dossier.show", $dossier->id) }}';
    });
});
</script>
@endsection